/***********************************************************************/
/*  This file is part of the ARM Toolchain package                     */
/*  Copyright KEIL ELEKTRONIK GmbH 2003 - 2010                         */
/***********************************************************************/
/*                                                                     */
/*  FlashDev.C:  Flash Programming Functions adapted                   */
/*               for ES32F3xx Flash                                    */
/*                                                                     */
/***********************************************************************/
#include "FlashOS.H"        /* FlashOS Structures */
#include "FlashParams.h"    /* flash parameters specification */
#include "ald_conf.h"
#include "nand.h"
#include "debugModule.h"


/*
 *  Initialize Flash Programming Functions
 *    Parameter:      adr:  Device Base Address
 *                    clk:  Clock Frequency (Hz)
 *                    fnc:  Function Code (1 - Erase, 2 - Program, 3 - Verify)
 *    Return Value:   0 - OK,  1 - Failed
 */
int Init(unsigned long adr, unsigned long clk, unsigned long fnc)
{
    /* select default HRC and init clock */
    ald_cmu_clock_config_default();
    ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_EBI, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_ECC, ENABLE);
    
    /* Init EBI device */
    ald_cmu_div_config(CMU_HCLK_2,CMU_DIV_1); 
    nand_init();
    
    /* feed independent wdt */
    ald_iwdt_feed_dog();
    
    #ifdef OutputDebugInfo
        Debug_Output_Init();
        DebugPrintf("Initialization Done\n");
    #endif
    
    return 0;
}

/*
 *  De-Initialize Flash Programming Functions
 *    Parameter:      fnc:  Function Code (1 - Erase, 2 - Program, 3 - Verify)
 *    Return Value:   0 - OK,  1 - Failed
 */
int UnInit(unsigned long fnc)
{
    #ifdef OutputDebugInfo
        DebugPrintf("DeInit Start\n");
    #endif

    /* feed independent wdt */
    ald_iwdt_feed_dog();

    #ifdef OutputDebugInfo
        DebugPrintf("DeInit Done\n");
    #endif

    return 0;
}

/*
 *  Erase complete Flash Memory
 *    Return Value:   0 - OK,  1 - Failed
 */

int EraseChip(void) 
{
    unsigned long i = 0UL;

    #ifdef OutputDebugInfo
        DebugPrintf("Chip Erase Start\n");
    #endif

    nand_pin_init();

    for (; i < FLASH_SIZE; i += FLASH_SECTORE_SIZE)
    {
        nand_address_t __addr;

        nand_addr_get(i, &__addr);
        flm_nand_erase_block(&g_nand_handle, &__addr);
        
        /* feed independent wdt */
        ald_iwdt_feed_dog();
    }

    #ifdef OutputDebugInfo
        DebugPrintf("Chip Erase Done\n");
    #endif

    return 0;
}

/*
 *  Erase Sector in Flash Memory
 *    Parameter:      adr:  Sector Address
 *    Return Value:   0 - OK,  1 - Failed
 */
int EraseSector(unsigned long adr)
{
    nand_address_t __addr;

    #ifdef OutputDebugInfo
        DebugPrintf("Sector Erase Start, address: 0x%x\n", adr - FLASH_BASE_ADDR);
    #endif

    nand_pin_init();
    nand_addr_get(adr - FLASH_BASE_ADDR, &__addr);
    flm_nand_erase_block(&g_nand_handle, &__addr);

    /* feed independent wdt */
    ald_iwdt_feed_dog();

    #ifdef OutputDebugInfo
        DebugPrintf("Sector Erase Done\n");
    #endif

    return 0;
}

/*
 *  Program Page in Flash Memory
 *    Parameter:      adr:  Page Start Address
 *                    sz:   Page Size
 *                    buf:  Page Data
 *    Return Value:   0 - OK,  1 - Failed
 */
int ProgramPage(unsigned long adr, unsigned long sz, unsigned char *buf)
{
    nand_address_t __addr;

    #ifdef OutputDebugInfo
    DebugPrintf("Page Program Start, start address: 0x%X, data length : 0x%x\n", adr - FLASH_BASE_ADDR, sz);
    #endif

    #ifdef OutputDebugInfo_Detail
    while (sz != 0)
    {
        DebugPrintf("Content at offset %d in Hex: %.2X\n", sz, *(buf + sz));
        sz--;
    }
    #endif

    /* Init Pins for NAND Flash */
    nand_pin_init();
    /* Get address */
    nand_addr_get(adr - FLASH_BASE_ADDR, &__addr);
    /* prgram page */
    if (sz > NAND_PAGE_SIZE)
    {
        flm_nand_write_page_8b(&g_nand_handle, &__addr, buf, 1);
        flm_nand_write_sparearea_8b(&g_nand_handle, &__addr, buf + NAND_PAGE_SIZE, 1);
    }
    else
    {
        flm_nand_write_page_8b(&g_nand_handle, &__addr, buf, 1);
    }

    /* feed independent wdt */
    ald_iwdt_feed_dog();

    #ifdef OutputDebugInfo
        DebugPrintf("Page Program Done\n");
    #endif

    return 0;
}

 /*
  *  Verify Flash Contents
  *    Parameter:      adr:  Start Address
  *                    sz:   Size (in bytes)
  *                    buf:  Data
  *    Return Value:   (adr+sz) - OK, Failed Address
  */
unsigned long Verify(unsigned long adr, unsigned long sz, unsigned char *buf)
{
    nand_address_t __addr;
    unsigned int byteIndex;
    unsigned int pageIndex;
    unsigned int flashPageNum;
    unsigned long dataLength;
    unsigned char CmpBuffer[FLASH_PAGE_SIZE];

    /* Init Pins for NAND Flash */
    nand_pin_init();
    /* Get address */
    nand_addr_get(adr - FLASH_BASE_ADDR, &__addr);
    /* get data in buf and compare */
    dataLength = sz;
    flashPageNum = sz / FLASH_PAGE_SIZE + 1;
    for (pageIndex = 0; pageIndex < flashPageNum; pageIndex++)
    {
        if (dataLength < NAND_PAGE_SIZE)
        {
            flm_nand_read_page_8b(&g_nand_handle, &__addr, CmpBuffer, 1);
            for (byteIndex = 0; byteIndex < dataLength; byteIndex++)
            {
                if (CmpBuffer[byteIndex] != buf[pageIndex * FLASH_PAGE_SIZE + byteIndex])
                    return (pageIndex * FLASH_PAGE_SIZE + byteIndex);
            }
        }
        else
        {
            flm_nand_read_page_8b(&g_nand_handle, &__addr, CmpBuffer, 1);
            flm_nand_read_sparearea_8b(&g_nand_handle, &__addr, CmpBuffer + NAND_PAGE_SIZE, 1);
            for (byteIndex = 0; byteIndex < ((dataLength > FLASH_PAGE_SIZE) ? FLASH_PAGE_SIZE : dataLength); byteIndex++)
            {
                if (CmpBuffer[byteIndex] != buf[pageIndex * FLASH_PAGE_SIZE + byteIndex])
                    return (pageIndex * FLASH_PAGE_SIZE + byteIndex);
            }
            
            (dataLength < FLASH_PAGE_SIZE) ? (dataLength -= 0): (dataLength -= FLASH_PAGE_SIZE);
        }
    }

    #ifdef OutputDebugInfo
        DebugPrintf("Verify ...\n");
    #endif

    return (adr + sz);
}


int BlankCheck(unsigned long adr, unsigned long sz, unsigned char pat)
{
    #ifdef OutputDebugInfo
        DebugPrintf("Blank Check Done\n");
    #endif

    return 1;                                        /* Always Force Erase */
}
